const BATCH_COUNT = 20;
const SCROLL_THRESHOLD = 0.2 * _getHeight();

// PREVIEW and EXPORT are defined in HTML head
const weekdays = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

let pageNum = 0;

function getNavigationStack() {
    const stack = sessionStorage.getItem('navigationStack');
    return stack ? JSON.parse(stack) : [];
}

function saveNavigationStack(stack) {
    sessionStorage.setItem('navigationStack', JSON.stringify(stack));
}

function pushNavigationStack(page, scrollPosition, messageId) {
    const stack = getNavigationStack();
    // Check if the page and messageId combo already exists in the stack
    var isDuplicate = stack.some(function(entry) {
        return entry.page === page && entry.messageId === messageId;
    });
    // Only push if it's not a duplicate
    if (!isDuplicate) {
        stack.push({ page, scrollPosition, messageId });
        saveNavigationStack(stack);
    }
}

function popNavigationStack() {
    const stack = getNavigationStack();
    const position = stack.pop();
    saveNavigationStack(stack);
    return position;
}

/*
 * Java connector
 */
const jsConnector = {
    showChat: function(displayName) {
        showChat(displayName);
    },
    loadMessages: function(msgDivs) {
        loadMessages(msgDivs);
    },
    setNoMoreMessages: function() {
        setNoMoreMessages();
    }
};

function getJsConnector() {
    return jsConnector;
}

function getJavaConnector() {
    if (window.javaConnector) {
        return window.javaConnector;
    } else {
        if (!isCBS()) {
            console.error('Java connector not defined!');
        }
        return null;
    }
}

document.addEventListener('DOMContentLoaded', function () {
    const mediaElements = document.querySelectorAll('source[src^="Shared Media/"], img[src^="Shared Media/"], video[src^="Shared Media/"]');
    mediaElements.forEach(element => {
        const originalSrc = element.getAttribute('src');
        const parts = originalSrc.split('/');
        const resName = parts[1];
        const mediaFile = window.sharedMedias[resName];

        if (mediaFile) {
            const newSrc = `Shared Media/${mediaFile.shortName}/${mediaFile.displayName}`;
            if (element.tagName.toLowerCase() === 'source') {
                const parentElement = element.closest('audio') || element.closest('video');
                if (parentElement) {
                    const type = element.getAttribute('type');
                    const newParent = document.createElement(parentElement.tagName.toLowerCase());
                    newParent.setAttribute('controls', '');
                    const newSource = document.createElement('source');
                    newSource.setAttribute('src', newSrc);
                    newSource.setAttribute('type', type);
                    newParent.appendChild(newSource);
                    parentElement.parentNode.replaceChild(newParent, parentElement);
                    newParent.load();
                }
            } else if (element.tagName.toLowerCase() === 'img') {
                element.setAttribute('src', newSrc);
                const onclick = element.getAttribute('onclick');
                if (onclick && onclick.includes(originalSrc)) {
                    const newOnclick = onclick.replace(originalSrc, newSrc);
                    element.setAttribute('onclick', newOnclick);
                }
            }
        } else {
            console.warn(`No mediaFile found for resName: ${resName}`);
            element.setAttribute('src', originalSrc);
        }
    });

    const quotedElements = document.querySelectorAll('.quoted');
    quotedElements.forEach(element => {
        element.addEventListener('click', function (event) {
            event.preventDefault();

            const onclickAttr = element.getAttribute('onclick');
            const targetUrlMatch = onclickAttr.match(/location\.href\s*=\s*`([^`]+)`/);
            const targetUrl = targetUrlMatch ? targetUrlMatch[1] : '';

            const currentPage = window.location.pathname;
            const chatBody = document.getElementById('chat-body');
            const scrollPosition = chatBody.scrollTop;

            const messageContainer = element.closest('.message-container');
            const currentMessageId = messageContainer ? messageContainer.id : '';
            pushNavigationStack(currentPage, scrollPosition, currentMessageId);

            if (targetUrl) {
                window.location.href = targetUrl;
            }
        });
    });

    let chatBody = document.getElementById('chat-body');
    if (PREVIEW) {
        chatBody.addEventListener('scroll', function (event) {
            let scroll = chatBody.scrollTop;
            if (scroll <= SCROLL_THRESHOLD) {
                logDebug('JS: Lazy loading');
                // Lazy loading
                _loadNextPage();
            }
        });
        _loadNextPage();
        // Scroll to bottom initially
        chatBody.scrollTop = chatBody.scrollHeight;
    }
    if (EXPORT) {
        processMessageTimestamps();
        const stack = getNavigationStack();
        if (stack.length > 0) {
            const returnButton = document.getElementById('return-button');
            if (returnButton) {
                returnButton.classList.add('visible');
                returnButton.addEventListener('click', function () {
                    const position = popNavigationStack();
                    if (position) {
                        window.location.href = position.page + (position.messageId ? `#${position.messageId}` : '');
                    }
                    const updatedStack = getNavigationStack();
                    if (updatedStack.length === 0) {
                        returnButton.classList.remove('visible');
                    }
                });
            }
        }
    }
});

function addReturnButton(messageContent) {
    const stack = getNavigationStack();
    if (stack.length === 0) return;

    const existingButtons = messageContent.querySelectorAll('.return-button');
    existingButtons.forEach(button => button.remove());

    const position = stack[stack.length - 1];
    const { page, scrollPosition, messageId } = position;

    const returnButton = document.createElement('button');
    returnButton.textContent = 'Return to Previous Position';
    returnButton.className = 'return-button';
    returnButton.style.position = 'absolute';
    returnButton.style.bottom = '-30px';
    returnButton.style.right = '2px';
    returnButton.style.padding = '5px 10px';
    returnButton.style.backgroundColor = '#007bff';
    returnButton.style.color = 'white';
    returnButton.style.border = 'none';
    returnButton.style.borderRadius = '3px';
    returnButton.style.cursor = 'pointer';
    returnButton.style.fontSize = '12px';

    returnButton.addEventListener('click', function () {
        returnButton.remove();

        const position = popNavigationStack();
        if (position) {
            if (position.messageId) {
                window.location.href = `${position.page}#${position.messageId}`;
            } else {
                window.location.href = position.page;
                localStorage.setItem('scrollToPosition', position.scrollPosition);
            }
        }

        const updatedStack = getNavigationStack();
        if (updatedStack.length === 0) {
            sessionStorage.removeItem('navigationStack');
            window.parent.postMessage({ action: 'navigationStackCleared' }, '*');
        }
    });

    messageContent.style.position = 'relative';
    messageContent.appendChild(returnButton);
}

function _loadNextPage() {
    logDebug('JS: _loadNextPage, pageNum=' + pageNum);
    if (pageNum < 0) { // Reached the end of chat
        return;
    }
    let javaConnector = getJavaConnector();
    if (javaConnector) {
        javaConnector.loadMessages(pageNum * BATCH_COUNT, BATCH_COUNT);
    }
}

function quickDownload(sharedFile, name, displayName, fallbackUrl) {
    logDebug('JS: quickDownloadOrOpenUrl, sharedFile=' + sharedFile + ', name=' + name + ', displayName=' + displayName + ', fallbackUrl=' + fallbackUrl);
    let javaConnector = getJavaConnector();
    if (javaConnector) {
        javaConnector.quickDownloadOrOpenUrl(sharedFile, name, displayName, fallbackUrl);
    } else if (isCBS()) {
        if (typeof quickDownloadOrOpenUrl === "function") {
            quickDownloadOrOpenUrl(sharedFile, name, displayName, fallbackUrl);
        }
    }
}

function openUrl(url) {
    logDebug('JS: openUrl, url=' + url);
    let javaConnector = getJavaConnector();
    if (javaConnector) {
        javaConnector.openUrl(url);
    } else {
        window.open(url, '_blank');
    }
}

/*
 * From Java
 */
function showChat(displayName) {
    let chatBody = document.getElementById('chat-body');
    chatBody.innerHTML = '';
    pageNum = 0;
    let chatHeader = document.getElementById('chat-header');
    chatHeader.innerHTML = `
        <div class='space-title'>
            <div class='space-name'>
                <span>${displayName}</span>
            </div>
        </div>
    `;
    // Do not replace document title for cbs preview
    if (!isCBS()) {
        document.title = displayName;
    }
    _loadNextPage();
}

function loadMessages(msgDivs) {
    if (msgDivs) {
        logDebug('JS: loadMessages');
        let chatBody = document.getElementById('chat-body');
        let prevHeight = chatBody.scrollHeight;
        chatBody.insertAdjacentHTML('afterbegin', msgDivs);
        processMessageTimestamps();
        chatBody.scrollTop = chatBody.scrollHeight - prevHeight;
        pageNum++;
    }
}

function setNoMoreMessages() {
    pageNum = -1;
}

/*
 * Export only functions
 */

function openMediaFile(resName) {
    if (sharedMedias && sharedMedias[resName]) {
        const mediaFile = sharedMedias[resName];
        let localMediaUrl = 'Shared Media/' + mediaFile.shortName + '/' + encodeFilename(mediaFile.displayName);
        safeOpenFile(localMediaUrl);
    } else {
        alert('Media file not found');
    }
}

function openDriveFile(driveFileId, fallbackUrl) {
    if (sharedFiles) {
        let relativePath = sharedFiles[driveFileId];
        if (relativePath) {
            safeOpenFile(encodeFilename(relativePath));
            return;
        }
    }
    openUrl(fallbackUrl);
}

/*
 * UI functionalities
 */
function processMessageTimestamps() {
    let chatBody = document.getElementById('chat-body');
    if (PREVIEW) {
        let systemMessage = document.querySelector('.system-message');
        if (systemMessage) {
            systemMessage.remove();
        }
    }
    let lastCreateDateStr = '';
    let sentAts = document.querySelectorAll('.sent-at[msg-create-time]');
    for (let i = 0; i < sentAts.length; i++) {
        let sentAt = sentAts[i];
        let createTime = sentAt.getAttribute('msg-create-time');
        let displayTime = sentAt.getAttribute('msg-display-time');
        if (!displayTime) {
            displayTime = createTime;
        }
        let msgState = sentAt.getAttribute('msg-state');
        let displayDate = new Date(displayTime);
        let displayMonth = months[displayDate.getMonth()];
        let displayDay = displayDate.getDate();
        let displayHour = String(displayDate.getHours()).padStart(2, '0');
        let displayMin = String(displayDate.getMinutes()).padStart(2, '0');

        let innerHtml = '';
        if (msgState === 'deleted') {
            innerHtml += 'Deleted at ';
        }
        innerHtml += `${displayMonth} ${displayDay}, ${displayHour}:${displayMin}`;
        if (msgState === 'edited') {
            innerHtml += ' <strong>·</strong> Edited';
        }
        sentAt.innerHTML = innerHtml;
        if (EXPORT || i > 0) {
            sentAt.removeAttribute('msg-create-time');
            sentAt.removeAttribute('msg-display-time');
        }
        if (msgState) {
            sentAt.removeAttribute('msg-state');
        }
        let createDate = new Date(createTime);
        let createWeekday = weekdays[createDate.getDay()];
        let createMonth = months[createDate.getMonth()];
        let createDay = createDate.getDate();
        let createYear = createDate.getFullYear();
        let createDateStr = `${createWeekday}, ${createMonth} ${createDay}, ${createYear}`;
        if (i === 0 || createDateStr !== lastCreateDateStr) {
            let systemMsg = document.createElement('div');
            systemMsg.className = 'system-message';
            systemMsg.textContent = createDateStr;
            sentAt.parentElement.parentElement.insertAdjacentElement('beforebegin', systemMsg);
            lastCreateDateStr = createDateStr;
        }
    }
}

function handleHashChange() {
    let hash = location.hash;
    if (hash) {
        let messageId = hash.substring(1);
        let target = document.getElementById(messageId);

        if (target) {
            let messageContent = target.querySelector('.message-content');
            if (messageContent) {
                messageContent.classList.add('highlight');
                setTimeout(function () {
                    messageContent.classList.remove('highlight');
                }, 1000);
                addReturnButton(messageContent);
            }
            target.scrollIntoView({ behavior: 'smooth' });
        }
    }else {
        const scrollToPosition = localStorage.getItem('scrollToPosition');
        if (scrollToPosition) {
            const chatBody = document.getElementById('chat-body');
            chatBody.scrollTop = parseInt(scrollToPosition, 10);
            localStorage.removeItem('scrollToPosition');
        }
    }
}

window.onload = handleHashChange;
window.addEventListener('hashchange', handleHashChange);

/*
 * Misc
 */
function logDebug(msg) {
    let javaConnector = getJavaConnector();
    if (javaConnector) {
        javaConnector.logDebug(msg);
    }
}

// from teams.js
function _getHeight() {
    let iHeight = 0;
    if (typeof(window.innerWidth) == 'number') {
        // Non-IE
        iHeight = window.innerHeight;
    } else if (document.documentElement && (document.documentElement.clientWidth || document.documentElement.clientHeight)) {
        // IE 6+ in 'standards compliant mode'
        iHeight = document.documentElement.clientHeight;
    } else if (document.body && (document.body.clientWidth || document.body.clientHeight)) {
        // IE 4 compatible
        iHeight = document.body.clientHeight;
    }
    return iHeight;
}

function isCBS() {
    return (typeof CBS !== 'undefined' && CBS);
}